<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Account;
use App\Models\User;
  
class AccountController extends Controller
{
    

    public function create()
    {
        $accounts = Account::all();

        // Get users who don't have a petty cash account assigned
        $usersWithPettyCash = Account::whereNotNull('user_id')->pluck('user_id')->toArray();
        $users = User::whereNotIn('id', $usersWithPettyCash)->get();

        // Find the Petty Cash parent account
        $pettyCashAccount = Account::where('code', '1004')->first();
        $pettyCashAccountId = $pettyCashAccount ? $pettyCashAccount->id : null;

        return view('admin.accounts.create', compact('accounts', 'users', 'pettyCashAccountId'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|unique:accounts,code',
            'name' => 'required',
            'type' => 'required|in:asset,liability,equity,income,expense,cogs',
            'parent_id' => 'nullable|exists:accounts,id',
            'user_id' => 'nullable|exists:users,id',
            'is_posting' => 'required',
        ]);
        $validated['is_posting'] = $request->has('is_posting');

        Account::create($validated);
        return redirect()->route('admin.accounts.index')->with('success', 'Account created successfully.');
    }
    public function index()
    {
        // Fetch accounts, for example:
        $accounts = Account::orderBy('code', 'asc')->get();
        return view('admin.accounts.index', compact('accounts'));
    }

    public function edit(Account $account)
    {
        $accounts = Account::where('id', '!=', $account->id)->get(); // exclude self

        // Get users who don't have a petty cash account assigned (excluding current account's user)
        $usersWithPettyCash = Account::whereNotNull('user_id')
            ->where('id', '!=', $account->id)
            ->pluck('user_id')->toArray();
        $users = User::whereNotIn('id', $usersWithPettyCash)->get();

        // Find the Petty Cash parent account
        $pettyCashAccount = Account::where('code', '1004')->first();
        $pettyCashAccountId = $pettyCashAccount ? $pettyCashAccount->id : null;

        return view('admin.accounts.edit', compact('account', 'accounts', 'users', 'pettyCashAccountId'));
    }

    public function update(Request $request, Account $account)
    {
        $validated = $request->validate([
            'code' => 'required|string|max:255',
            'name' => 'required|string|max:255',
            'type' => 'required|in:asset,liability,equity,income,expense,cogs',
            'parent_id' => 'nullable|exists:accounts,id',
            'user_id' => 'nullable|exists:users,id',
            'is_posting' => 'nullable|boolean',
        ]);

        $validated['is_posting'] = $request->has('is_posting');

        $account->update($validated);

        return redirect()->route('admin.accounts.index')->with('success', 'Account updated successfully.');
    }

    public function destroy($id)
    {
        $account = Account::findOrFail($id);

        // Prevent delete if child accounts exist
        if ($account->children()->exists()) {
            return redirect()->route('accounts.index')
                ->with('error', 'Cannot delete account with child accounts.');
        }

        // Prevent delete if journal transactions exist
        if ($account->journalTransactions()->exists()) {
            return redirect()->route('accounts.index')
                ->with('error', 'Cannot delete account linked to journal transactions.');
        }

        $account->delete();

        return redirect()->route('admin.accounts.index')
            ->with('success', 'Account deleted successfully.');
    }


}
