<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\PettyCashExpense;
use App\Models\JournalEntry;
use App\Models\JournalTransaction;

class PettyCashApprovalController extends Controller
{
    public function index()
    {
        $expenses = PettyCashExpense::where('status', 'pending')
            ->with(['user', 'expenseAccount', 'pettyCashAccount'])
            ->orderBy('created_at')
            ->get();

        return view('admin.pettycash.approvals', compact('expenses'));
    }

    public function show(PettyCashExpense $expense)
    {
        $expense->load(['user', 'expenseAccount', 'pettyCashAccount']);

        // Get all expense accounts for dropdown
        $expenseAccounts = \App\Models\Account::where('type', 'expense')
            ->where('is_posting', 1)
            ->orderBy('name')
            ->get();

        return view('admin.pettycash.show-expense', compact('expense', 'expenseAccounts'));
    }

    public function update(Request $request, PettyCashExpense $expense)
    {
        // Only allow editing if expense is still pending
        if ($expense->status !== 'pending') {
            return back()->withErrors(['error' => 'Cannot edit an expense that has already been processed.']);
        }

        $request->validate([
            'expense_account_id' => 'required|exists:accounts,id',
            'description' => 'required|string|max:255',
        ]);

        $expense->update([
            'expense_account_id' => $request->expense_account_id,
            'description' => $request->description,
        ]);

        return back()->with('success', 'Expense details updated successfully');
    }

    public function approve(Request $request, PettyCashExpense $expense)
    {
        // Validate vat_claimable input if expense has VAT
        if ($expense->is_vat_inclusive && $expense->vat_amount > 0) {
            $request->validate([
                'vat_claimable' => 'nullable|boolean',
            ]);
        }

        DB::transaction(function () use ($request, $expense) {

            // Update vat_claimable status if VAT is present
            if ($expense->is_vat_inclusive && $expense->vat_amount > 0) {
                $expense->vat_claimable = $request->has('vat_claimable');
            }

            $journal = JournalEntry::create([
                'date' => $expense->expense_date,
                'description' => 'Petty cash expense: ' . $expense->description,
                'user_id' => $expense->user_id,
            ]);

            // Calculate amounts
            $vatAmount = $expense->vat_amount;
            $isVatClaimable = $expense->vat_claimable ?? false;

            // Determine expense amount based on VAT claimability
            if ($vatAmount > 0 && $isVatClaimable) {
                // VAT is claimable - expense is amount excluding VAT
                $expenseAmount = $expense->amount_excluding_vat;
            } else {
                // No VAT or VAT not claimable - expense is total amount
                $expenseAmount = $expense->amount;
            }

            // Debit expense account
            JournalTransaction::create([
                'journal_entry_id' => $journal->id,
                'account_id' => $expense->expense_account_id,
                'debit' => $expenseAmount,
                'credit' => 0,
                'user_id' => $expense->user_id,
                'description' => $isVatClaimable && $vatAmount > 0 ? 'Expense (excl. claimable VAT)' : null,
            ]);

            // If VAT is claimable, debit VAT Receivable account
            if ($vatAmount > 0 && $isVatClaimable) {
                $vatReceivableAccount = \App\Models\Account::where('code', '1201')->first();

                if ($vatReceivableAccount) {
                    JournalTransaction::create([
                        'journal_entry_id' => $journal->id,
                        'account_id' => $vatReceivableAccount->id,
                        'debit' => $vatAmount,
                        'credit' => 0,
                        'user_id' => $expense->user_id,
                        'description' => 'VAT on purchase (claimable)',
                    ]);
                }
            }

            // Credit petty cash (total amount paid including VAT)
            JournalTransaction::create([
                'journal_entry_id' => $journal->id,
                'account_id' => $expense->petty_cash_account_id,
                'debit' => 0,
                'credit' => $expense->amount,
                'user_id' => $expense->user_id,
            ]);

            $expense->update([
                'status' => 'approved',
                'approved_by' => auth()->id(),
                'approved_at' => now(),
                'vat_claimable' => $isVatClaimable,
            ]);
        });

        return back()->with('success', 'Expense approved and posted to ledger');
    }

    public function reject(Request $request, PettyCashExpense $expense)
    {
        $request->validate([
            'reason' => 'required|string|max:255',
        ]);

        $expense->update([
            'status' => 'rejected',
            'approved_by' => auth()->id(),
            'rejection_reason' => $request->reason,
        ]);

        return back()->with('success', 'Expense rejected');
    }
}
