# Laravel Project Documentation

## Project Overview
RMS (Resource Management System) is a comprehensive business management application built with Laravel. It handles accounting operations including journal entries, ledger management, invoicing, customer/vendor management, equipment tracking, and petty cash management with granular role-based permissions.

**Laravel Version:** 12.12.0
**PHP Version:** 8.2.12
**Database:** SQLite (default), MySQL (optional)

## Project Structure

### Key Directories
```
app/
├── Http/
│   ├── Controllers/
│   │   ├── Admin/           # Admin area controllers
│   │   ├── Auth/            # Authentication controllers
│   │   └── User/            # User area controllers
│   ├── Middleware/          # Custom middleware
│   └── Requests/            # Form request validation
└── Models/                  # Eloquent models

resources/
├── views/
│   ├── admin/               # Admin panel views
│   ├── auth/                # Authentication views
│   ├── layouts/             # Layout templates
│   └── user/                # User dashboard views
├── js/                      # JavaScript files
└── css/                     # Stylesheets

routes/
├── web.php                  # Web routes (main routing file)
├── api.php                  # API routes
└── console.php              # Artisan commands

database/
├── migrations/              # Database migrations
├── seeders/                 # Database seeders
└── factories/               # Model factories

tests/
├── Feature/                 # Feature tests
└── Unit/                    # Unit tests
```

## Architecture & Patterns

### Design Patterns Used
- [x] MVC (Model-View-Controller)
- [ ] Repository Pattern
- [ ] Service Layer Pattern (recommended for complex business logic)
- [ ] Observer Pattern (Events/Listeners)
- [x] Factory Pattern (for testing)

### Key Conventions
- **Naming**: Follow PSR standards and Laravel conventions
  - Controllers: `CustomerController`, `InvoiceController` (singular, PascalCase)
  - Models: `User`, `Customer`, `Invoice` (singular, PascalCase)
  - Tables: `users`, `customers`, `invoices` (plural, snake_case)
  - Migrations: `create_users_table`, `add_status_to_invoices_table`
  - Routes: Named routes with prefix notation (e.g., `admin.invoices.index`)

- **Code Organization**:
  - Admin controllers in `App\Http\Controllers\Admin` namespace
  - User controllers in `App\Http\Controllers\User` namespace
  - Controllers should be thin, delegating to services
  - Use Form Requests for validation

## Database Schema

### Core Modules

#### Accounting System
```
accounts
- id
- account_number (unique)
- account_name
- account_type (asset/liability/equity/revenue/expense)
- is_posting (boolean - whether account accepts direct postings)
- balance
- created_at
- updated_at

journal_entries
- id
- entry_number (unique)
- date
- description
- reference
- created_by
- created_at
- updated_at

journal_transactions
- id
- journal_entry_id
- account_id
- debit
- credit
- created_at
- updated_at
```

#### Customer & Invoicing
```
customers
- id
- name
- email
- phone
- address
- created_at
- updated_at

customer_contacts
- id
- customer_id
- name
- email
- phone
- position
- created_at
- updated_at

invoices
- id
- invoice_number
- customer_id
- subject
- date
- due_date
- status (draft/sent/paid/overdue)
- subtotal
- tax
- total
- journal_entry_id (for accounting integration)
- journal_posted_at
- created_at
- updated_at

invoice_items
- id
- invoice_id
- item_id
- description
- quantity
- unit_price
- amount
- created_at
- updated_at

items
- id
- name
- description
- unit_price
- created_at
- updated_at
```

#### Vendor Management
```
vendors
- id
- name
- email
- phone
- address
- created_at
- updated_at

vendor_contacts
- id
- vendor_id
- name
- email
- phone
- position
- created_at
- updated_at
```

#### Equipment & Assets
```
equipment
- id
- name
- description
- serial_number
- purchase_date
- purchase_price
- vendor_id
- created_at
- updated_at
```

#### Petty Cash Management
```
petty_cash_expenses
- id
- user_id
- account_id (petty cash account)
- amount
- description
- receipt_photo
- status (pending/approved/rejected)
- approved_by
- approved_at
- rejection_reason
- created_at
- updated_at
```

#### Quotes
```
quotes
- id
- customer_id
- quote_number
- date
- valid_until
- status
- subtotal
- tax
- total
- created_at
- updated_at
```

### Model Relationships
- User hasMany JournalEntry (created_by)
- Account hasMany JournalTransaction
- JournalEntry hasMany JournalTransaction
- Customer hasMany Invoice
- Customer hasMany Quote
- Customer hasMany CustomerContact
- Invoice belongsTo Customer
- Invoice hasMany InvoiceItem
- Invoice belongsTo JournalEntry (optional, when posted)
- InvoiceItem belongsTo Item
- Vendor hasMany Equipment
- Vendor hasMany VendorContact
- PettyCashExpense belongsTo User
- PettyCashExpense belongsTo Account
- PettyCashExpense belongsTo User (approver)

## Authentication & Authorization

**Authentication:** Laravel UI (traditional session-based authentication)
**Authorization:** Spatie Laravel Permission package (roles & permissions)

### User Roles
- **Admin** - Full system access
- **User** - Limited access based on specific permissions
- **Accountant** - Can manage journals, ledger, and approve expenses
- **Staff** - Can submit petty cash expenses, view own data

### Key Permissions
```
# Dashboard
- view dashboard

# Accounting
- manage accounts
- view journals
- create journals
- edit journals
- delete journals
- create expenses
- view ledger

# Petty Cash
- view petty cash (admin view)
- view own petty cash
- add petty cash expense
- topup petty cash
- approve petty cash

# Business Operations
- view invoices
- manage invoices
- manage customers
- manage vendors
- manage items
- manage quotes
- manage equipment
```

## Application Features

### Accounting Module
- Double-entry accounting system
- Chart of accounts with account types
- Journal entry creation with multi-line transactions
- Automatic balance calculations
- Ledger generation with date ranges
- Integration with invoice posting

### Invoice Management
- Create and manage customer invoices
- Line item management
- Status tracking (draft/sent/paid/overdue)
- Automatic journal entry creation when posted
- Customer contact management

### Petty Cash System
- Employee expense submission
- Receipt photo upload
- Approval workflow (pending → approved/rejected)
- Account-based petty cash tracking
- Top-up functionality for petty cash accounts
- Personal ledger view for employees

### Customer & Vendor Management
- Customer profiles with contact management
- Vendor profiles with contact management
- Equipment tracking linked to vendors
- Quote generation for customers

## Frontend Stack

**Framework:** Blade (Laravel's templating engine)
**CSS Framework:** Bootstrap 5 + Tailwind CSS 4
**Admin Template:** AdminLTE 3 (via jeroennoten/laravel-adminlte)
**Build Tool:** Vite 6

### Admin Panel
- AdminLTE-based responsive admin interface
- Role-based navigation and access control
- Dashboard with key metrics
- Sidebar navigation with permission-based menu items

## Testing

**Testing Framework:** PHPUnit 11.5
**Testing Database:** SQLite (in-memory for testing)

### Running Tests
```bash
# Run all tests
php artisan test

# Run specific test
php artisan test --filter CustomerTest

# Run with coverage
php artisan test --coverage
```

## Environment Configuration

### Required Environment Variables
```
APP_NAME="RMS"
APP_ENV=local
APP_KEY=base64:...
APP_DEBUG=true
APP_URL=http://localhost

# Database (SQLite default)
DB_CONNECTION=sqlite

# Or MySQL
# DB_CONNECTION=mysql
# DB_HOST=127.0.0.1
# DB_PORT=3306
# DB_DATABASE=rms
# DB_USERNAME=root
# DB_PASSWORD=

# Session
SESSION_DRIVER=database
SESSION_LIFETIME=120

# Queue
QUEUE_CONNECTION=database

# Cache
CACHE_STORE=database

# Mail
MAIL_MAILER=log
MAIL_FROM_ADDRESS="hello@example.com"
MAIL_FROM_NAME="${APP_NAME}"

# Filesystem
FILESYSTEM_DISK=local
```

## Third-Party Packages

- [x] Spatie Laravel Permission - Role and permission management
- [x] Laravel AdminLTE - Admin panel theme
- [x] Laravel UI - Authentication scaffolding
- [x] Doctrine DBAL - Database abstraction (for migrations)
- [x] Laravel Pint - Code style fixer
- [ ] Payment Gateway integration
- [ ] AWS S3 for file storage (receipt photos)
- [ ] Email service (for invoice sending)

## Development Workflow

### Setup Instructions
```bash
# Clone repository
git clone [repository-url]
cd rms

# Install dependencies
composer install
npm install

# Environment setup
cp .env.example .env
php artisan key:generate

# Database setup (SQLite default)
touch database/database.sqlite
php artisan migrate
php artisan db:seed

# Install Spatie permissions
php artisan vendor:publish --provider="Spatie\Permission\PermissionServiceProvider"

# Build frontend assets
npm run build

# Start development server (runs all services concurrently)
composer run dev
# OR manually:
php artisan serve
npm run dev
```

### Running Development Servers
```bash
# Option 1: Use composer script (runs everything concurrently)
composer run dev
# This runs: server, queue worker, log viewer (pail), and vite

# Option 2: Manual separate terminals
php artisan serve          # Terminal 1
php artisan queue:work     # Terminal 2
npm run dev                # Terminal 3
```

### Branch Strategy
- `main` - Production
- `develop` - Development
- `feature/*` - New features
- `bugfix/*` - Bug fixes

## Code Style & Standards

**PHP:** Follow PSR-12 coding standards
**Linting:** [e.g., PHP CS Fixer, Laravel Pint]

### Running Code Style Fixes
```bash
./vendor/bin/pint
# or
php artisan pint
```

## Performance Considerations

- [x] Database indexing on foreign keys
- [ ] Query optimization (N+1 prevention using eager loading)
- [x] Queue jobs for long-running tasks (configured)
- [x] Database session driver for scalability
- [ ] Redis caching for production
- [ ] Database query caching for reports
- [ ] Implement lazy loading for large datasets

### Optimization Tips
- Use `->with()` for eager loading relationships in controllers
- Index frequently queried columns (invoice_number, account_number, etc.)
- Consider caching ledger reports and account balances
- Use database transactions for journal entries

## Security Considerations

- [x] CSRF protection enabled (Laravel default)
- [x] SQL injection prevention via Eloquent ORM
- [x] XSS protection via Blade `{{ }}` escaping
- [x] Password hashing using bcrypt
- [x] Role-based access control (Spatie Permission)
- [x] Permission-based route protection
- [ ] Rate limiting on API routes
- [ ] File upload validation (receipt photos)
- [ ] Input sanitization for financial data
- [ ] Audit logging for financial transactions

### Permission Best Practices
- Always use middleware for route protection
- Check permissions in controllers for fine-grained access
- Never expose sensitive financial data without permission checks
- Implement audit trails for journal entries and approvals

## Useful Artisan Commands

```bash
# Generate files
php artisan make:model Invoice -mfc
php artisan make:controller Admin/ReportController --resource
php artisan make:request StoreInvoiceRequest
php artisan make:migration add_field_to_table

# Database
php artisan migrate
php artisan migrate:fresh --seed
php artisan db:seed --class=PermissionSeeder

# Cache management
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
php artisan optimize        # Cache config, routes, and views

# Queue management
php artisan queue:work
php artisan queue:restart
php artisan queue:failed    # View failed jobs

# Permission management
php artisan permission:cache-reset
php artisan permission:create-role admin
php artisan permission:create-permission "manage invoices"

# Code quality
php artisan pint            # Fix code style
php artisan test            # Run tests
```

## Known Issues & Technical Debt

### Areas for Improvement
- Add service layer for complex business logic (especially accounting calculations)
- Implement repository pattern for data access
- Add comprehensive test coverage
- Implement audit logging for all financial transactions
- Add data export functionality (Excel, PDF for invoices and reports)
- Implement email notifications for invoice status changes
- Add invoice payment tracking and reconciliation
- Create dashboard widgets with real-time financial metrics
- Implement backup and disaster recovery procedures

### Known Issues
- Duplicate controller files in `app/Http/Controllers/Admin/1/` directory (cleanup needed)
- Missing API endpoints (currently web-only application)
- No soft deletes on financial records (consider implementing)

## Application Routes

### Public Routes
- `/` - Welcome page
- `/login` - Login page
- `/logout` - Logout (POST)
- `/home` - User home dashboard (authenticated)

### Admin Routes (prefix: `/admin`)
All require authentication and specific permissions:

- `/admin/dashboard` - Admin dashboard
- `/admin/accounts` - Chart of accounts management
- `/admin/journals` - Journal entry management
- `/admin/journals/record-expense` - Quick expense recording
- `/admin/ledger` - General ledger view
- `/admin/invoices` - Invoice management
- `/admin/customers` - Customer management
- `/admin/vendors` - Vendor management
- `/admin/items` - Item catalog
- `/admin/quotes` - Quote management
- `/admin/equipment` - Equipment tracking
- `/admin/petty-cash` - Petty cash admin view
- `/admin/pettycash/my-ledger` - Personal petty cash ledger
- `/admin/pettycash/my-expense` - Submit petty cash expense
- `/admin/pettycash/approvals` - Approve/reject expenses

### User Routes (prefix: `/user`)
- `/user/dashboard` - User dashboard (role:user)

## Resources & Documentation

- [Laravel 12 Documentation](https://laravel.com/docs/12.x)
- [Spatie Permission Docs](https://spatie.be/docs/laravel-permission)
- [AdminLTE Documentation](https://adminlte.io/docs)
- [Vite Documentation](https://vitejs.dev/)
- [Bootstrap 5 Docs](https://getbootstrap.com/docs/5.2/)

## Deployment Checklist

### Pre-Deployment
```bash
# Run tests
composer run test

# Fix code style
./vendor/bin/pint

# Check for security vulnerabilities
composer audit

# Optimize autoloader
composer install --optimize-autoloader --no-dev
```

### Deployment Steps
```bash
# Pull latest code
git pull origin main

# Install dependencies (production)
composer install --no-dev --optimize-autoloader

# Run migrations
php artisan migrate --force

# Clear and cache configuration
php artisan config:cache
php artisan route:cache
php artisan view:cache

# Build frontend assets
npm ci
npm run build

# Restart queue workers
php artisan queue:restart

# Set proper permissions
chmod -R 755 storage bootstrap/cache
```

### Environment Setup (Production)
- Set `APP_ENV=production`
- Set `APP_DEBUG=false`
- Generate new `APP_KEY`
- Configure production database credentials
- Set up queue worker as systemd service
- Configure scheduled tasks in cron
- Enable HTTPS and force SSL
- Set up log rotation

## AI Assistant Guidelines

When helping with this project:
1. **Follow Laravel best practices and conventions**
2. **Use type hints and return types in PHP 8.2+**
3. **Write descriptive commit messages**
4. **Include PHPDoc blocks for complex methods**
5. **Suggest tests for new features**
6. **Consider performance implications** (especially for financial calculations)
7. **Follow the existing code style** (use Laravel Pint)
8. **Always check permissions** before allowing access to sensitive data
9. **Maintain double-entry accounting principles** for all financial transactions
10. **Validate financial data rigorously** (amounts, dates, balances)
11. **Consider audit trails** for financial operations
12. **Test thoroughly** before deploying accounting-related changes

### Accounting-Specific Guidelines
- Always ensure journal entries balance (total debits = total credits)
- Never delete posted journal entries (consider void/reverse functionality)
- Maintain referential integrity between invoices and journal entries
- Validate account types before posting transactions
- Use database transactions for multi-step financial operations
- Round monetary values to 2 decimal places consistently
- Consider tax implications in calculations

### Permission Guidelines
- Always use named permissions (not roles) for route protection
- Check permissions at both route and controller levels
- Never bypass permission checks for "convenience"
- Document all permission requirements in controller methods
- Test permission-based access thoroughly
