<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Auth\LoginController;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\AdminController;
use App\Http\Controllers\UserController;

// Admin Controllers
use App\Http\Controllers\Admin\{
    AccountController,
    JournalEntryController,
    JournalExpenseController,
    LedgerController,
    InvoiceController,
    InvoiceItemController,
    CustomerController,
    CustomerContactController,
    VendorController,
    ItemController,
    QuoteController,
    EquipmentController,
    PettyCashController,
    PettyCashApprovalController,
    UserController as AdminUserController,
    RoleController,
    PermissionController
};

// HR Controllers
use App\Http\Controllers\Admin\HR\{
    EmployeeController,
    DepartmentController,
    DesignationController,
    AttendanceController,
    LeaveApplicationController
};

/*
|--------------------------------------------------------------------------
| Public
|--------------------------------------------------------------------------
*/
Route::get('/', function () {
    return view('welcome');
});

/*
|--------------------------------------------------------------------------
| Authentication
|--------------------------------------------------------------------------
*/
Route::middleware('guest')->group(function () {
    Route::get('login', [LoginController::class, 'showLoginForm'])->name('login');
    Route::post('login', [LoginController::class, 'login']);
});

Route::post('logout', [LoginController::class, 'logout'])
    ->middleware('auth')
    ->name('logout');

Route::get('/home', [HomeController::class, 'index'])
    ->middleware('auth')
    ->name('home');

/*
|--------------------------------------------------------------------------
| Admin Area (shared for admin + permitted users)
|--------------------------------------------------------------------------
| IMPORTANT:
| - No role middleware here
| - Permissions control access
|--------------------------------------------------------------------------
*/
Route::middleware(['auth'])
    ->prefix('admin')
    ->as('admin.')
    ->group(function () {

        /*
        |--------------------------------------------------------------------------
        | Dashboard
        |--------------------------------------------------------------------------
        */
        Route::get('/dashboard', [AdminController::class, 'dashboard'])
            ->middleware('permission:view dashboard')
            ->name('dashboard');

        /*
        |--------------------------------------------------------------------------
        | User Management
        |--------------------------------------------------------------------------
        */
        Route::resource('users', AdminUserController::class)
            ->middleware('permission:manage users');

        /*
        |--------------------------------------------------------------------------
        | Roles & Permissions
        |--------------------------------------------------------------------------
        */
        Route::resource('roles', RoleController::class)
            ->middleware('permission:manage users');

        Route::resource('permissions', PermissionController::class)
            ->except(['show', 'edit', 'update'])
            ->middleware('permission:manage users');

        /*
        |--------------------------------------------------------------------------
        | Accounts
        |--------------------------------------------------------------------------
        */
        Route::resource('accounts', AccountController::class)
            ->middleware('permission:manage accounts');

        /*
        |--------------------------------------------------------------------------
        | Petty Cash - Admin
        |--------------------------------------------------------------------------
        */
        Route::get('petty-cash', [PettyCashController::class, 'index'])
            ->middleware('permission:view petty cash')
            ->name('pettycash.index');

        Route::get('petty-cash/{account}', [PettyCashController::class, 'show'])
            ->middleware('permission:view petty cash')
            ->name('pettycash.show');

        Route::get('petty-cash/{account}/topup', [PettyCashController::class, 'topupForm'])
            ->middleware('permission:topup petty cash')
            ->name('pettycash.topup');

        Route::post('petty-cash/{account}/topup', [PettyCashController::class, 'topupStore'])
            ->middleware('permission:topup petty cash')
            ->name('pettycash.topup.store');

        /*
        |--------------------------------------------------------------------------
        | Petty Cash - User
        |--------------------------------------------------------------------------
        */
        Route::get('pettycash/my-ledger', [PettyCashController::class, 'myLedger'])
            ->middleware('permission:view own petty cash')
            ->name('pettycash.my-ledger');

        Route::get('pettycash/my-expense', [PettyCashController::class, 'myExpenseForm'])
            ->middleware('permission:add petty cash expense')
            ->name('pettycash.my-expense');

        Route::post('pettycash/my-expense', [PettyCashController::class, 'storeMyExpense'])
            ->middleware('permission:add petty cash expense')
            ->name('pettycash.my-expense.store');

        Route::get('pettycash/my-expense/{expense}/edit', [PettyCashController::class, 'editMyExpense'])
            ->middleware('permission:add petty cash expense')
            ->name('pettycash.my-expense.edit');

        Route::put('pettycash/my-expense/{expense}', [PettyCashController::class, 'updateMyExpense'])
            ->middleware('permission:add petty cash expense')
            ->name('pettycash.my-expense.update');

        // Petty Cash - Expense Account Settings (Admin only)
        Route::get('pettycash/settings/expense-accounts', [PettyCashController::class, 'expenseAccountSettings'])
            ->middleware('permission:manage accounts')
            ->name('pettycash.expense-account-settings');

        Route::put('pettycash/settings/expense-accounts', [PettyCashController::class, 'updateExpenseAccountSettings'])
            ->middleware('permission:manage accounts')
            ->name('pettycash.expense-account-settings.update');

        /*
        |--------------------------------------------------------------------------
        | Petty Cash - Approvals (Admin / Accountant)
        |--------------------------------------------------------------------------
        */
        Route::middleware('permission:approve petty cash')->group(function () {

            Route::get('pettycash/approvals',
                [PettyCashApprovalController::class, 'index']
            )->name('pettycash.approvals.index');

            Route::get('pettycash/approvals/{expense}',
                [PettyCashApprovalController::class, 'show']
            )->name('pettycash.approvals.show');

            Route::put('pettycash/approvals/{expense}',
                [PettyCashApprovalController::class, 'update']
            )->name('pettycash.approvals.update');

            Route::post('pettycash/approvals/{expense}/approve',
                [PettyCashApprovalController::class, 'approve']
            )->name('pettycash.approvals.approve');

            Route::post('pettycash/approvals/{expense}/reject',
                [PettyCashApprovalController::class, 'reject']
            )->name('pettycash.approvals.reject');

        });

        /*
        |--------------------------------------------------------------------------
        | Journals
        |--------------------------------------------------------------------------
        */
        Route::get('journals', [JournalEntryController::class, 'index'])
            ->middleware('permission:view journals')
            ->name('journals.index');

        Route::get('journals/create', [JournalEntryController::class, 'create'])
            ->middleware('permission:create journals')
            ->name('journals.create');

        Route::post('journals', [JournalEntryController::class, 'store'])
            ->middleware('permission:create journals')
            ->name('journals.store');

        Route::get('journals/{journal}/edit', [JournalEntryController::class, 'edit'])
            ->middleware('permission:edit journals')
            ->name('journals.edit');

        Route::put('journals/{journal}', [JournalEntryController::class, 'update'])
            ->middleware('permission:edit journals')
            ->name('journals.update');

        Route::delete('journals/{journal}', [JournalEntryController::class, 'destroy'])
            ->middleware('permission:delete journals')
            ->name('journals.destroy');

        /*
        |--------------------------------------------------------------------------
        | Record Expense (Special Journal)
        |--------------------------------------------------------------------------
        */
        Route::get('journals/record-expense', [JournalExpenseController::class, 'create'])
            ->middleware('permission:create expenses')
            ->name('journals.record-expense');

        Route::post('journals/record-expense', [JournalExpenseController::class, 'store'])
            ->middleware('permission:create expenses')
            ->name('journals.record-expense.store');

        /*
        |--------------------------------------------------------------------------
        | Ledger
        |--------------------------------------------------------------------------
        */
        Route::get('ledger', [LedgerController::class, 'index'])
            ->middleware('permission:view ledger')
            ->name('ledger.index');

        Route::post('ledger', [LedgerController::class, 'generate'])
            ->middleware('permission:view ledger')
            ->name('ledger.generate');

        /*
        |--------------------------------------------------------------------------
        | Invoices
        |--------------------------------------------------------------------------
        */
        Route::resource('invoices', InvoiceController::class)
            ->middleware('permission:view invoices');

        Route::resource('invoice_items', InvoiceItemController::class)
            ->middleware('permission:manage invoices');

        /*
        |--------------------------------------------------------------------------
        | Master Data
        |--------------------------------------------------------------------------
        */
        Route::resource('customers', CustomerController::class)
            ->middleware('permission:manage customers');

        Route::resource('customer_contacts', CustomerContactController::class)
            ->middleware('permission:manage customers');

        Route::resource('vendors', VendorController::class)
            ->middleware('permission:manage vendors');

        Route::resource('items', ItemController::class)
            ->middleware('permission:manage items');

        Route::resource('quotes', QuoteController::class)
            ->middleware('permission:manage quotes');

        Route::resource('equipment', EquipmentController::class)
            ->middleware('permission:manage equipment');

        /*
        |--------------------------------------------------------------------------
        | HR Module Routes
        |--------------------------------------------------------------------------
        */

        // Departments
        Route::resource('hr/departments', DepartmentController::class)
            ->names('hr.departments')
            ->middleware('permission:manage departments');

        // Designations
        Route::resource('hr/designations', DesignationController::class)
            ->names('hr.designations')
            ->middleware('permission:manage designations');

        // Employees
        Route::resource('hr/employees', EmployeeController::class)
            ->names('hr.employees')
            ->middleware('permission:manage employees');

        // Create User for Employee
        Route::get('hr/employees/{employee}/create-user', [EmployeeController::class, 'createUserForm'])
            ->middleware('permission:manage employees')
            ->name('hr.employees.create-user');
        Route::post('hr/employees/{employee}/create-user', [EmployeeController::class, 'createUser'])
            ->middleware('permission:manage employees')
            ->name('hr.employees.store-user');

        // Attendance
        Route::prefix('hr/attendance')->name('hr.attendance.')->group(function () {
            Route::get('/', [AttendanceController::class, 'index'])
                ->middleware('permission:view attendance')
                ->name('index');
            Route::get('/mark', [AttendanceController::class, 'markForm'])
                ->middleware('permission:mark own attendance')
                ->name('mark');
            Route::get('/check', [AttendanceController::class, 'checkAttendance'])
                ->middleware('permission:mark own attendance')
                ->name('check');
            Route::post('/mark', [AttendanceController::class, 'store'])
                ->middleware('permission:mark own attendance')
                ->name('store');
            Route::get('/approve', [AttendanceController::class, 'approvalList'])
                ->middleware('permission:approve attendance')
                ->name('approve');
            Route::post('/{attendance}/approve', [AttendanceController::class, 'approve'])
                ->middleware('permission:approve attendance')
                ->name('approve.submit');
        });

        // Leave Applications
        Route::prefix('hr/leaves')->name('hr.leaves.')->group(function () {
            Route::get('/', [LeaveApplicationController::class, 'index'])
                ->middleware('permission:view own leaves')
                ->name('index');
            Route::get('/create', [LeaveApplicationController::class, 'create'])
                ->middleware('permission:apply leave')
                ->name('create');
            Route::post('/', [LeaveApplicationController::class, 'store'])
                ->middleware('permission:apply leave')
                ->name('store');
            Route::get('/all', [LeaveApplicationController::class, 'all'])
                ->middleware('permission:view all leaves')
                ->name('all');
            Route::get('/{leave}', [LeaveApplicationController::class, 'show'])
                ->name('show');
            Route::post('/{leave}/manager-approve', [LeaveApplicationController::class, 'managerApprove'])
                ->middleware('permission:approve leave as manager')
                ->name('manager-approve');
            Route::post('/{leave}/hr-approve', [LeaveApplicationController::class, 'hrApprove'])
                ->middleware('permission:approve leave as hr')
                ->name('hr-approve');
            Route::post('/{leave}/reject', [LeaveApplicationController::class, 'reject'])
                ->middleware('permission:reject leave')
                ->name('reject');
        });
    });

/*
|--------------------------------------------------------------------------
| User Area (optional dashboard only)
|--------------------------------------------------------------------------
*/
Route::middleware(['auth', 'role:user'])
    ->prefix('user')
    ->as('user.')
    ->group(function () {
        Route::get('/dashboard', [UserController::class, 'dashboard'])
            ->name('dashboard');
    });
